'use strict';

var assert = require('assert');
var search = require('./search');

describe('Search assets', function () {
  it('returns nothing if nothing is passed', function () {
    assert.deepEqual(search.assets(), []);
  });

  it('returns matches based on a pattern', function () {
    var modules = {
      'foobar.txt': {},
      'barbar.css': {}
    };
    var extensions = ['.txt'];
    var matches = [{ name: 'foobar.txt', asset: {} }];

    assert.deepEqual(search.assets(modules, extensions), matches);
  });

  it('returns matches if they have query', function () {
    var modules = {
      'foobar.txt?123': {},
      'barbar.css': {}
    };
    var extensions = ['.txt'];
    var matches = [{ name: 'foobar.txt?123', asset: {} }];

    assert.deepEqual(search.assets(modules, extensions), matches);
  });
});

describe('Search files', function () {
  it('returns nothing if nothing is passed', function () {
    assert.deepEqual(search.files(), []);
  });

  it('returns matches based on extension', function () {
    var modules = ['foobar.txt', 'barbar.css'];
    var extensions = ['.txt'];
    var matches = ['foobar.txt'];

    assert.deepEqual(search.files(modules, extensions), matches);
  });

  it('does not fail with missing modules', function () {
    var modules = ['foobar.txt', '', 'barbar.css'];
    var extensions = ['.txt'];
    var matches = ['foobar.txt'];

    assert.deepEqual(search.files(modules, extensions), matches);
  });

  it('returns matches based on extension with a customized getter', function () {
    var modules = {
      foobar: {
        resource: 'foobar.txt'
      },
      barbar: {
        resource: 'barbar.css'
      }
    };
    var extensions = ['.txt'];
    var matches = ['foobar.txt'];

    assert.deepEqual(search.files(modules, extensions, function (file) {
      return file.resource;
    }), matches);
  });

  it('does not fail with missing modules when a getter fails', function () {
    var modules = {
      foobar: {
        resource: 'foobar.txt'
      },
      demo: {},
      barbar: {
        resource: 'barbar.css'
      }
    };
    var extensions = ['.txt'];
    var matches = ['foobar.txt'];

    assert.deepEqual(search.files(modules, extensions, function (file) {
      return file.resource;
    }), matches);
  });
});