'use strict';

var assert = require('assert');
var parse = require('./parse');

describe('Parse entry paths', function () {
  it('returns an empty array by default', function () {
    assert.deepEqual(parse.entryPaths(), []);
  });

  it('returns an object as itself', function () {
    var o = { a: ['a', 'b', 'c'] };

    assert.deepEqual(parse.entryPaths(o), o);
  });

  it('puts a string inside an array', function () {
    var str = 'foobar';

    assert.deepEqual(parse.entryPaths(str), [str]);
  });
});

describe('Flatten entry paths', function () {
  it('returns an array as itself', function () {
    var a = ['a', 'b', 'c'];

    assert.deepEqual(parse.flatten(a), a);
  });

  it('returns an object of arrays as one flat array', function () {
    var o = { a: ['a', 'b'], b: ['c', 'd'] };

    assert.deepEqual(parse.flatten(o), ['a', 'b', 'c', 'd']);
  });
});

describe('Parse entries', function () {
  it('returns paths if there is no chunk name', function () {
    var paths = ['a', 'b', 'c'];

    assert.deepEqual(parse.entries(paths), paths);
  });

  it('returns paths if paths are an array already', function () {
    var paths = ['a', 'b', 'c'];

    assert.deepEqual(parse.entries(paths, 'foobar'), paths);
  });

  it('returns chunk paths', function () {
    var entryPaths = ['a', 'b', 'c'];
    var paths = {
      foobar: entryPaths
    };

    assert.deepEqual(parse.entries(paths, 'foobar'), entryPaths);
  });

  it('returns chunk path wrapped in an array', function () {
    var entryPaths = 'a';
    var paths = {
      foobar: entryPaths
    };

    assert.deepEqual(parse.entries(paths, 'foobar'), [entryPaths]);
  });

  it('returns an empty array if failed to find entry', function () {
    var paths = {
      foobar: 'a'
    };

    assert.deepEqual(parse.entries(paths, 'barbar'), []);
  });

  it('returns an empty array if failed to find entry with multiple paths', function () {
    var paths = {
      foobar: 'a',
      barbar: 'b'
    };

    assert.deepEqual(parse.entries(paths, 'foofoo'), []);
  });
});